"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityCCCommandsSupportedGet = exports.SecurityCCCommandsSupportedReport = exports.SecurityCCNetworkKeySet = exports.SecurityCCNetworkKeyVerify = exports.SecurityCCSchemeInherit = exports.SecurityCCSchemeGet = exports.SecurityCCSchemeReport = exports.SecurityCCCommandEncapsulationNonceGet = exports.SecurityCCCommandEncapsulation = exports.SecurityCCNonceGet = exports.SecurityCCNonceReport = exports.SecurityCC = exports.SecurityCCAPI = exports.SecurityCommand = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const crypto_1 = require("crypto");
const SendDataMessages_1 = require("../controller/SendDataMessages");
const Constants_1 = require("../message/Constants");
const API_1 = require("./API");
const CommandClass_1 = require("./CommandClass");
// @noSetValueAPI This is an encapsulation CC
// All the supported commands
var SecurityCommand;
(function (SecurityCommand) {
    SecurityCommand[SecurityCommand["CommandsSupportedGet"] = 2] = "CommandsSupportedGet";
    SecurityCommand[SecurityCommand["CommandsSupportedReport"] = 3] = "CommandsSupportedReport";
    SecurityCommand[SecurityCommand["SchemeGet"] = 4] = "SchemeGet";
    SecurityCommand[SecurityCommand["SchemeReport"] = 5] = "SchemeReport";
    SecurityCommand[SecurityCommand["SchemeInherit"] = 8] = "SchemeInherit";
    SecurityCommand[SecurityCommand["NetworkKeySet"] = 6] = "NetworkKeySet";
    SecurityCommand[SecurityCommand["NetworkKeyVerify"] = 7] = "NetworkKeyVerify";
    SecurityCommand[SecurityCommand["NonceGet"] = 64] = "NonceGet";
    SecurityCommand[SecurityCommand["NonceReport"] = 128] = "NonceReport";
    SecurityCommand[SecurityCommand["CommandEncapsulation"] = 129] = "CommandEncapsulation";
    SecurityCommand[SecurityCommand["CommandEncapsulationNonceGet"] = 193] = "CommandEncapsulationNonceGet";
})(SecurityCommand = exports.SecurityCommand || (exports.SecurityCommand = {}));
function getAuthenticationData(senderNonce, receiverNonce, ccCommand, sendingNodeId, receivingNodeId, encryptedPayload) {
    return Buffer.concat([
        senderNonce,
        receiverNonce,
        Buffer.from([
            ccCommand,
            sendingNodeId,
            receivingNodeId,
            encryptedPayload.length,
        ]),
        encryptedPayload,
    ]);
}
const HALF_NONCE_SIZE = 8;
// TODO: Ignore commands if received via multicast
let SecurityCCAPI = class SecurityCCAPI extends API_1.PhysicalCCAPI {
    supportsCommand(_cmd) {
        // All commands are mandatory
        return true;
    }
    async sendEncapsulated(encapsulated, requestNextNonce = false) {
        if (requestNextNonce) {
            this.assertSupportsCommand(SecurityCommand, SecurityCommand.CommandEncapsulation);
        }
        else {
            this.assertSupportsCommand(SecurityCommand, SecurityCommand.CommandEncapsulationNonceGet);
        }
        const cc = new (requestNextNonce
            ? SecurityCCCommandEncapsulationNonceGet
            : SecurityCCCommandEncapsulation)(this.driver, {
            nodeId: this.endpoint.nodeId,
            encapsulated,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    /**
     * Requests a new nonce for Security CC encapsulation
     */
    async getNonce(options = {}) {
        this.assertSupportsCommand(SecurityCommand, SecurityCommand.NonceGet);
        const { standalone = false, storeAsFreeNonce = false } = options;
        const cc = new SecurityCCNonceGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, {
            ...this.commandOptions,
            // Standalone nonce requests must be handled immediately
            priority: standalone
                ? Constants_1.MessagePriority.Normal
                : Constants_1.MessagePriority.PreTransmitHandshake,
            // Only try getting a nonce once
            maxSendAttempts: 1,
            // We don't want failures causing us to treat the node as asleep or dead
            // The "real" transaction will do that for us
            changeNodeStatusOnMissingACK: standalone,
        });
        if (!response)
            return;
        const nonce = response.nonce;
        if (storeAsFreeNonce) {
            const secMan = this.driver.securityManager;
            secMan.setNonce({
                issuer: this.endpoint.nodeId,
                nonceId: secMan.getNonceId(nonce),
            }, { nonce, receiver: this.driver.controller.ownNodeId }, { free: true });
        }
        return nonce;
    }
    /**
     * Responds to a NonceGet request. The message is sent without any retransmission etc.
     * The return value indicates whether a nonce was successfully sent
     */
    async sendNonce() {
        this.assertSupportsCommand(SecurityCommand, SecurityCommand.NonceReport);
        if (!this.driver.securityManager) {
            throw new core_1.ZWaveError(`Nonces can only be sent if secure communication is set up!`, core_1.ZWaveErrorCodes.Driver_NoSecurity);
        }
        const nonce = this.driver.securityManager.generateNonce(this.endpoint.nodeId, HALF_NONCE_SIZE);
        const nonceId = this.driver.securityManager.getNonceId(nonce);
        const cc = new SecurityCCNonceReport(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            nonce,
        });
        const msg = new SendDataMessages_1.SendDataRequest(this.driver, {
            command: cc,
            // Seems we need these options or some nodes won't accept the nonce
            transmitOptions: SendDataMessages_1.TransmitOptions.ACK | SendDataMessages_1.TransmitOptions.AutoRoute,
        });
        try {
            await this.driver.sendMessage(msg, {
                ...this.commandOptions,
                priority: Constants_1.MessagePriority.Handshake,
            });
        }
        catch (e) {
            if (e instanceof core_1.ZWaveError &&
                (e.code === core_1.ZWaveErrorCodes.Controller_ResponseNOK ||
                    e.code === core_1.ZWaveErrorCodes.Controller_CallbackNOK ||
                    e.code === core_1.ZWaveErrorCodes.Controller_NodeTimeout ||
                    e.code === core_1.ZWaveErrorCodes.Controller_MessageDropped)) {
                // The nonce could not be sent, invalidate it
                this.driver.securityManager.deleteNonce(nonceId);
                return false;
            }
            else {
                // Pass other errors through
                throw e;
            }
        }
        return true;
    }
    async getSecurityScheme() {
        this.assertSupportsCommand(SecurityCommand, SecurityCommand.SchemeGet);
        const cc = new SecurityCCSchemeGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
        // There is only one scheme, so we hardcode it
        return [0];
    }
    async inheritSecurityScheme() {
        this.assertSupportsCommand(SecurityCommand, SecurityCommand.SchemeInherit);
        const cc = new SecurityCCSchemeInherit(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
        // There is only one scheme, so we don't return anything here
    }
    async setNetworkKey(networkKey) {
        this.assertSupportsCommand(SecurityCommand, SecurityCommand.NetworkKeySet);
        const keySet = new SecurityCCNetworkKeySet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            networkKey,
        });
        const cc = new SecurityCCCommandEncapsulation(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            encapsulated: keySet,
            alternativeNetworkKey: Buffer.alloc(16, 0),
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getSupportedCommands() {
        this.assertSupportsCommand(SecurityCommand, SecurityCommand.CommandsSupportedGet);
        const cc = new SecurityCCCommandsSupportedGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, ["supportedCCs", "controlledCCs"]);
        }
    }
};
SecurityCCAPI = __decorate([
    CommandClass_1.API(core_1.CommandClasses.Security)
], SecurityCCAPI);
exports.SecurityCCAPI = SecurityCCAPI;
let SecurityCC = class SecurityCC extends CommandClass_1.CommandClass {
    async interview(complete = true) {
        const node = this.getNode();
        const endpoint = this.getEndpoint();
        const api = endpoint.commandClasses.Security.withOptions({
            priority: Constants_1.MessagePriority.NodeQuery,
        });
        // This only needs to be done once
        if (complete) {
            this.driver.controllerLog.logNode(node.id, {
                message: "querying securely supported commands...",
                direction: "outbound",
            });
            const resp = await api.getSupportedCommands();
            if (!resp) {
                if (node.isSecure === true) {
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: "Querying securely supported commands timed out, skipping Security interview...",
                        level: "warn",
                    });
                    // TODO: Abort interview?
                }
                else {
                    // We didn't know if the node was secure, assume that it is not actually included securely
                    this.driver.controllerLog.logNode(node.id, `The node is not included securely. Continuing interview non-securely.`);
                    node.isSecure = false;
                }
                return;
            }
            const logLines = [
                "received secure commands",
                "supported CCs:",
            ];
            for (const cc of resp.supportedCCs) {
                logLines.push(`· ${core_1.getCCName(cc)}`);
            }
            logLines.push("controlled CCs:");
            for (const cc of resp.controlledCCs) {
                logLines.push(`· ${core_1.getCCName(cc)}`);
            }
            this.driver.controllerLog.logNode(node.id, {
                message: logLines.join("\n"),
                direction: "inbound",
            });
            // Remember which commands are supported securely
            for (const cc of resp.supportedCCs) {
                endpoint.addCC(cc, {
                    isSupported: true,
                    secure: true,
                });
            }
            for (const cc of resp.controlledCCs) {
                endpoint.addCC(cc, {
                    isControlled: true,
                    secure: true,
                });
            }
            // We know for sure that the node is included securely
            if (node.isSecure !== true) {
                node.isSecure = true;
                this.driver.controllerLog.logNode(node.id, `The node is included securely.`);
            }
            // Remember that the interview is complete
            this.interviewComplete = true;
        }
    }
    /** Tests if a should be sent secure and thus requires encapsulation */
    static requiresEncapsulation(cc) {
        return (cc.secure &&
            // Already encapsulated (SecurityCCCommandEncapsulationNonceGet is a subclass)
            !(cc instanceof SecurityCCCommandEncapsulation) &&
            // Cannot be sent encapsulated
            !(cc instanceof SecurityCCNonceGet) &&
            !(cc instanceof SecurityCCNonceReport) &&
            !(cc instanceof SecurityCCSchemeGet) &&
            !(cc instanceof SecurityCCSchemeReport));
    }
    /** Encapsulates a command that should be sent encrypted */
    static encapsulate(driver, cc) {
        // TODO: When to return a SecurityCCCommandEncapsulationNonceGet?
        return new SecurityCCCommandEncapsulation(driver, {
            nodeId: cc.nodeId,
            encapsulated: cc,
        });
    }
};
SecurityCC = __decorate([
    CommandClass_1.commandClass(core_1.CommandClasses.Security),
    CommandClass_1.implementedVersion(1)
], SecurityCC);
exports.SecurityCC = SecurityCC;
let SecurityCCNonceReport = class SecurityCCNonceReport extends SecurityCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            core_1.validatePayload.withReason("Invalid nonce length")(this.payload.length === HALF_NONCE_SIZE);
            this.nonce = this.payload;
        }
        else {
            if (options.nonce.length !== HALF_NONCE_SIZE) {
                throw new core_1.ZWaveError(`Nonce must have length ${HALF_NONCE_SIZE}!`, core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            this.nonce = options.nonce;
        }
    }
    serialize() {
        this.payload = this.nonce;
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { nonce: shared_1.buffer2hex(this.nonce) },
        };
    }
};
SecurityCCNonceReport = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.NonceReport)
], SecurityCCNonceReport);
exports.SecurityCCNonceReport = SecurityCCNonceReport;
let SecurityCCNonceGet = class SecurityCCNonceGet extends SecurityCC {
};
SecurityCCNonceGet = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.NonceGet),
    CommandClass_1.expectedCCResponse(SecurityCCNonceReport)
], SecurityCCNonceGet);
exports.SecurityCCNonceGet = SecurityCCNonceGet;
function getCCResponseForCommandEncapsulation(sent) {
    if (sent.encapsulated.expectsCCResponse()) {
        return SecurityCCCommandEncapsulation;
    }
}
let SecurityCCCommandEncapsulation = class SecurityCCCommandEncapsulation extends SecurityCC {
    constructor(driver, options) {
        super(driver, options);
        if (!this.driver.controller.ownNodeId) {
            throw new core_1.ZWaveError(`Secure commands can can only be sent when the controller's node id is known!`, core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        else if (!this.driver.securityManager) {
            throw new core_1.ZWaveError(`Secure commands can only be sent when the network key for the driver is set`, core_1.ZWaveErrorCodes.Driver_NoSecurity);
        }
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // HALF_NONCE_SIZE bytes iv, 1 byte frame control, at least 1 CC byte, 1 byte nonce id, 8 bytes auth code
            core_1.validatePayload(this.payload.length >= HALF_NONCE_SIZE + 1 + 1 + 1 + 8);
            const iv = this.payload.slice(0, HALF_NONCE_SIZE);
            const encryptedPayload = this.payload.slice(HALF_NONCE_SIZE, -9);
            const nonceId = this.payload[this.payload.length - 9];
            const authCode = this.payload.slice(-8);
            // Retrieve the used nonce from the nonce store
            const nonce = this.driver.securityManager.getNonce(nonceId);
            // Only accept the message if the nonce hasn't expired
            core_1.validatePayload.withReason(`Nonce ${shared_1.num2hex(nonceId)} expired, cannot decode security encapsulated command.`)(!!nonce);
            // and mark the nonce as used
            this.driver.securityManager.deleteNonce(nonceId);
            this.authKey = this.driver.securityManager.authKey;
            this.encryptionKey = this.driver.securityManager.encryptionKey;
            // Validate the encrypted data
            const authData = getAuthenticationData(iv, nonce, this.ccCommand, this.nodeId, this.driver.controller.ownNodeId, encryptedPayload);
            const expectedAuthCode = core_1.computeMAC(authData, this.authKey);
            // Only accept messages with a correct auth code
            core_1.validatePayload.withReason("Invalid auth code, won't accept security encapsulated command.")(authCode.equals(expectedAuthCode));
            // Decrypt the encapsulated CC
            const frameControlAndDecryptedCC = core_1.decryptAES128OFB(encryptedPayload, this.encryptionKey, Buffer.concat([iv, nonce]));
            const frameControl = frameControlAndDecryptedCC[0];
            this.sequenceCounter = frameControl & 0b1111;
            this.sequenced = !!(frameControl & 16);
            this.secondFrame = !!(frameControl & 32);
            this.decryptedCCBytes = frameControlAndDecryptedCC.slice(1);
        }
        else {
            this.encapsulated = options.encapsulated;
            options.encapsulated.encapsulatingCC = this;
            if (options.alternativeNetworkKey) {
                this.authKey = core_1.generateAuthKey(options.alternativeNetworkKey);
                this.encryptionKey = core_1.generateEncryptionKey(options.alternativeNetworkKey);
            }
            else {
                this.authKey = this.driver.securityManager.authKey;
                this.encryptionKey = this.driver.securityManager.encryptionKey;
            }
        }
    }
    getPartialCCSessionId() {
        if (this.sequenced) {
            return {
                // Treat Encapsulation and EncapsulationNonceGet as one
                ccCommand: undefined,
                sequence: this.sequenceCounter,
            };
        }
        else {
            return {
                // Treat Encapsulation and EncapsulationNonceGet as one
                ccCommand: undefined,
            };
        }
    }
    expectMoreMessages() {
        return !!this.sequenced && !this.secondFrame;
    }
    mergePartialCCs(partials) {
        // Concat the CC buffers
        this.decryptedCCBytes = Buffer.concat([...partials, this].map((cc) => cc.decryptedCCBytes));
        // make sure this contains a complete CC command that's worth splitting
        core_1.validatePayload(this.decryptedCCBytes.length >= 2);
        // and deserialize the CC
        this.encapsulated = CommandClass_1.CommandClass.from(this.driver, {
            data: this.decryptedCCBytes,
            fromEncapsulation: true,
            encapCC: this,
        });
    }
    requiresPreTransmitHandshake() {
        // We require a new nonce if there is no free one,
        // we don't have one yet or if the old one has expired
        const secMan = this.driver.securityManager;
        // If the nonce is already known we don't need a handshake
        if (this.nonceId != undefined &&
            secMan.hasNonce({
                issuer: this.nodeId,
                nonceId: this.nonceId,
            })) {
            return false;
        }
        // Try to get a free nonce before requesting a new one
        const freeNonce = secMan.getFreeNonce(this.nodeId);
        if (freeNonce) {
            this.nonceId = secMan.getNonceId(freeNonce);
            return false;
        }
        return true;
    }
    async preTransmitHandshake() {
        // Request a nonce
        const nonce = await this.getNode().commandClasses.Security.getNonce();
        // TODO: Handle this more intelligent
        if (nonce) {
            // and store it
            const secMan = this.driver.securityManager;
            this.nonceId = secMan.getNonceId(nonce);
            secMan.setNonce({
                issuer: this.nodeId,
                nonceId: this.nonceId,
            }, { nonce, receiver: this.driver.controller.ownNodeId }, 
            // The nonce is reserved for this command
            { free: false });
        }
    }
    serialize() {
        function throwNoNonce() {
            throw new core_1.ZWaveError(`Security CC requires a nonce to be sent!`, core_1.ZWaveErrorCodes.SecurityCC_NoNonce);
        }
        // Try to find an active nonce
        if (this.nonceId == undefined)
            throwNoNonce();
        const receiverNonce = this.driver.securityManager.getNonce({
            issuer: this.nodeId,
            nonceId: this.nonceId,
        });
        if (!receiverNonce)
            throwNoNonce();
        // and mark it as used
        this.driver.securityManager.deleteNonce({
            issuer: this.nodeId,
            nonceId: this.nonceId,
        });
        const serializedCC = this.encapsulated.serialize();
        const plaintext = Buffer.concat([
            Buffer.from([0]),
            serializedCC,
        ]);
        // Encrypt the payload
        const senderNonce = crypto_1.randomBytes(HALF_NONCE_SIZE);
        const iv = Buffer.concat([senderNonce, receiverNonce]);
        const ciphertext = core_1.encryptAES128OFB(plaintext, this.encryptionKey, iv);
        // And generate the auth code
        const authData = getAuthenticationData(senderNonce, receiverNonce, this.ccCommand, this.driver.controller.ownNodeId, this.nodeId, ciphertext);
        const authCode = core_1.computeMAC(authData, this.authKey);
        this.payload = Buffer.concat([
            senderNonce,
            ciphertext,
            Buffer.from([this.nonceId]),
            authCode,
        ]);
        return super.serialize();
    }
    computeEncapsulationOverhead() {
        // Supervision CC adds 8 bytes IV, 1 byte frame control, 1 byte nonce ID, 8 bytes MAC
        return super.computeEncapsulationOverhead() + 18;
    }
    toLogEntry() {
        const message = {};
        if (this.nonceId != undefined) {
            message["nonce id"] = this.nonceId;
        }
        if (this.sequenced != undefined) {
            message.sequenced = this.sequenced;
            if (this.sequenced) {
                if (this.secondFrame != undefined) {
                    message["second frame"] = this.secondFrame;
                }
                if (this.sequenceCounter != undefined) {
                    message["sequence counter"] = this.sequenceCounter;
                }
            }
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
SecurityCCCommandEncapsulation = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.CommandEncapsulation),
    CommandClass_1.expectedCCResponse(getCCResponseForCommandEncapsulation, () => "checkEncapsulated")
], SecurityCCCommandEncapsulation);
exports.SecurityCCCommandEncapsulation = SecurityCCCommandEncapsulation;
// This is the same message, but with another CC command
let SecurityCCCommandEncapsulationNonceGet = class SecurityCCCommandEncapsulationNonceGet extends SecurityCCCommandEncapsulation {
};
SecurityCCCommandEncapsulationNonceGet = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.CommandEncapsulationNonceGet)
], SecurityCCCommandEncapsulationNonceGet);
exports.SecurityCCCommandEncapsulationNonceGet = SecurityCCCommandEncapsulationNonceGet;
let SecurityCCSchemeReport = class SecurityCCSchemeReport extends SecurityCC {
    // @noCCValues This CC has no values
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 1, 
        // Since it is unlikely that any more schemes will be added to S0, we hardcode the default scheme here (bit 0 = 0)
        (this.payload[0] & 0b1) === 0);
    }
};
SecurityCCSchemeReport = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.SchemeReport)
], SecurityCCSchemeReport);
exports.SecurityCCSchemeReport = SecurityCCSchemeReport;
let SecurityCCSchemeGet = class SecurityCCSchemeGet extends SecurityCC {
    constructor(driver, options) {
        super(driver, options);
        // Don't care, we won't get sent this and we have no options
    }
    serialize() {
        // Since it is unlikely that any more schemes will be added to S0, we hardcode the default scheme here (bit 0 = 0)
        this.payload = Buffer.from([0]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            // Hide the default payload line
            message: undefined,
        };
    }
};
SecurityCCSchemeGet = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.SchemeGet),
    CommandClass_1.expectedCCResponse(SecurityCCSchemeReport)
], SecurityCCSchemeGet);
exports.SecurityCCSchemeGet = SecurityCCSchemeGet;
let SecurityCCSchemeInherit = class SecurityCCSchemeInherit extends SecurityCC {
    constructor(driver, options) {
        super(driver, options);
        // Don't care, we won't get sent this and we have no options
    }
    serialize() {
        // Since it is unlikely that any more schemes will be added to S0, we hardcode the default scheme here (bit 0 = 0)
        this.payload = Buffer.from([0]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            // Hide the default payload line
            message: undefined,
        };
    }
};
SecurityCCSchemeInherit = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.SchemeInherit),
    CommandClass_1.expectedCCResponse(SecurityCCSchemeReport)
], SecurityCCSchemeInherit);
exports.SecurityCCSchemeInherit = SecurityCCSchemeInherit;
let SecurityCCNetworkKeyVerify = class SecurityCCNetworkKeyVerify extends SecurityCC {
};
SecurityCCNetworkKeyVerify = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.NetworkKeyVerify)
], SecurityCCNetworkKeyVerify);
exports.SecurityCCNetworkKeyVerify = SecurityCCNetworkKeyVerify;
let SecurityCCNetworkKeySet = class SecurityCCNetworkKeySet extends SecurityCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            if (options.networkKey.length !== 16) {
                throw new core_1.ZWaveError(`The network key must have length 16!`, core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            this.networkKey = options.networkKey;
        }
    }
    serialize() {
        this.payload = this.networkKey;
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "network key": shared_1.buffer2hex(this.networkKey) },
        };
    }
};
SecurityCCNetworkKeySet = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.NetworkKeySet),
    CommandClass_1.expectedCCResponse(SecurityCCNetworkKeyVerify)
], SecurityCCNetworkKeySet);
exports.SecurityCCNetworkKeySet = SecurityCCNetworkKeySet;
let SecurityCCCommandsSupportedReport = class SecurityCCCommandsSupportedReport extends SecurityCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 1);
        this.reportsToFollow = this.payload[0];
        const list = core_1.parseCCList(this.payload.slice(1));
        this._supportedCCs = list.supportedCCs;
        this._controlledCCs = list.controlledCCs;
    }
    getPartialCCSessionId() {
        // Nothing special we can distinguish sessions with
        return {};
    }
    expectMoreMessages() {
        return this.reportsToFollow > 0;
    }
    get supportedCCs() {
        return this._supportedCCs;
    }
    get controlledCCs() {
        return this._controlledCCs;
    }
    mergePartialCCs(partials) {
        // Concat the lists of CCs
        this._supportedCCs = [...partials, this]
            .map((report) => report._supportedCCs)
            .reduce((prev, cur) => prev.concat(...cur), []);
        this._controlledCCs = [...partials, this]
            .map((report) => report._controlledCCs)
            .reduce((prev, cur) => prev.concat(...cur), []);
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                reportsToFollow: this.reportsToFollow,
                supportedCCs: this._supportedCCs
                    .map((cc) => core_1.getCCName(cc))
                    .map((cc) => `\n· ${cc}`)
                    .join(""),
                controlledCCs: this._controlledCCs
                    .map((cc) => core_1.getCCName(cc))
                    .map((cc) => `\n· ${cc}`)
                    .join(""),
            },
        };
    }
};
SecurityCCCommandsSupportedReport = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.CommandsSupportedReport)
], SecurityCCCommandsSupportedReport);
exports.SecurityCCCommandsSupportedReport = SecurityCCCommandsSupportedReport;
let SecurityCCCommandsSupportedGet = class SecurityCCCommandsSupportedGet extends SecurityCC {
};
SecurityCCCommandsSupportedGet = __decorate([
    CommandClass_1.CCCommand(SecurityCommand.CommandsSupportedGet),
    CommandClass_1.expectedCCResponse(SecurityCCCommandsSupportedReport)
], SecurityCCCommandsSupportedGet);
exports.SecurityCCCommandsSupportedGet = SecurityCCCommandsSupportedGet;

//# sourceMappingURL=SecurityCC.js.map
