"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeCCTimeOffsetGet = exports.TimeCCTimeOffsetReport = exports.TimeCCTimeOffsetSet = exports.TimeCCDateGet = exports.TimeCCDateReport = exports.TimeCCTimeGet = exports.TimeCCTimeReport = exports.TimeCC = exports.TimeCCAPI = exports.TimeCommand = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const strings_1 = require("alcalzone-shared/strings");
const Constants_1 = require("../message/Constants");
const API_1 = require("./API");
const CommandClass_1 = require("./CommandClass");
// All the supported commands
var TimeCommand;
(function (TimeCommand) {
    TimeCommand[TimeCommand["TimeGet"] = 1] = "TimeGet";
    TimeCommand[TimeCommand["TimeReport"] = 2] = "TimeReport";
    TimeCommand[TimeCommand["DateGet"] = 3] = "DateGet";
    TimeCommand[TimeCommand["DateReport"] = 4] = "DateReport";
    TimeCommand[TimeCommand["TimeOffsetSet"] = 5] = "TimeOffsetSet";
    TimeCommand[TimeCommand["TimeOffsetGet"] = 6] = "TimeOffsetGet";
    TimeCommand[TimeCommand["TimeOffsetReport"] = 7] = "TimeOffsetReport";
})(TimeCommand = exports.TimeCommand || (exports.TimeCommand = {}));
// @noSetValueAPI
// Only the timezone information can be set and that accepts a non-primitive value
let TimeCCAPI = class TimeCCAPI extends API_1.CCAPI {
    supportsCommand(cmd) {
        switch (cmd) {
            case TimeCommand.TimeGet:
            case TimeCommand.DateGet:
                return this.isSinglecast(); // "mandatory"
            case TimeCommand.TimeOffsetGet:
                return this.version >= 2 && this.isSinglecast();
            case TimeCommand.TimeOffsetSet:
                return this.version >= 2;
        }
        return super.supportsCommand(cmd);
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getTime() {
        this.assertSupportsCommand(TimeCommand, TimeCommand.TimeGet);
        const cc = new TimeCCTimeGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, ["hour", "minute", "second"]);
        }
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getDate() {
        this.assertSupportsCommand(TimeCommand, TimeCommand.DateGet);
        const cc = new TimeCCDateGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, ["day", "month", "year"]);
        }
    }
    async setTimezone(timezone) {
        this.assertSupportsCommand(TimeCommand, TimeCommand.TimeOffsetSet);
        const cc = new TimeCCTimeOffsetSet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            standardOffset: timezone.standardOffset,
            dstOffset: timezone.dstOffset,
            dstStart: timezone.startDate,
            dstEnd: timezone.endDate,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    async getTimezone() {
        this.assertSupportsCommand(TimeCommand, TimeCommand.TimeOffsetGet);
        const cc = new TimeCCTimeOffsetGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return {
                standardOffset: response.standardOffset,
                dstOffset: response.dstOffset,
                startDate: response.dstStartDate,
                endDate: response.dstEndDate,
            };
        }
    }
};
TimeCCAPI = __decorate([
    CommandClass_1.API(core_1.CommandClasses.Time)
], TimeCCAPI);
exports.TimeCCAPI = TimeCCAPI;
let TimeCC = class TimeCC extends CommandClass_1.CommandClass {
    async interview(complete = true) {
        const node = this.getNode();
        const endpoint = this.getEndpoint();
        const api = endpoint.commandClasses.Time.withOptions({
            priority: Constants_1.MessagePriority.NodeQuery,
        });
        this.driver.controllerLog.logNode(node.id, {
            endpoint: this.endpointIndex,
            message: `${this.constructor.name}: doing a ${complete ? "complete" : "partial"} interview...`,
            direction: "none",
        });
        // Always keep the slave's time in sync
        if (api.version >= 2) {
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: "setting timezone information...",
                direction: "outbound",
            });
            // Set the correct timezone on this node
            const timezone = core_1.getDSTInfo() || core_1.getDefaultDSTInfo();
            await api.setTimezone(timezone);
        }
        // Remember that the interview is complete
        this.interviewComplete = true;
    }
};
TimeCC = __decorate([
    CommandClass_1.commandClass(core_1.CommandClasses.Time),
    CommandClass_1.implementedVersion(2)
], TimeCC);
exports.TimeCC = TimeCC;
let TimeCCTimeReport = class TimeCCTimeReport extends TimeCC {
    // @noCCValues Time is temporary :), we don't want to store that in a DB
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            core_1.validatePayload(this.payload.length >= 3);
            this.hour = this.payload[0] & 0b11111;
            core_1.validatePayload(this.hour >= 0, this.hour <= 23);
            this.minute = this.payload[1];
            core_1.validatePayload(this.minute >= 0, this.minute <= 59);
            this.second = this.payload[2];
            core_1.validatePayload(this.second >= 0, this.second <= 59);
        }
        else {
            this.hour = options.hour;
            this.minute = options.minute;
            this.second = options.second;
        }
    }
    serialize() {
        this.payload = Buffer.from([
            this.hour & 0b11111,
            this.minute,
            this.second,
        ]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                time: `${strings_1.padStart(this.hour.toString(), 2, "0")}:${strings_1.padStart(this.minute.toString(), 2, "0")}:${strings_1.padStart(this.second.toString(), 2, "0")}`,
            },
        };
    }
};
TimeCCTimeReport = __decorate([
    CommandClass_1.CCCommand(TimeCommand.TimeReport)
], TimeCCTimeReport);
exports.TimeCCTimeReport = TimeCCTimeReport;
let TimeCCTimeGet = class TimeCCTimeGet extends TimeCC {
};
TimeCCTimeGet = __decorate([
    CommandClass_1.CCCommand(TimeCommand.TimeGet),
    CommandClass_1.expectedCCResponse(TimeCCTimeReport)
], TimeCCTimeGet);
exports.TimeCCTimeGet = TimeCCTimeGet;
let TimeCCDateReport = class TimeCCDateReport extends TimeCC {
    // @noCCValues Time is temporary :), we don't want to store that in a DB
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            core_1.validatePayload(this.payload.length >= 4);
            this.year = this.payload.readUInt16BE(0);
            this.month = this.payload[2];
            this.day = this.payload[3];
        }
        else {
            this.year = options.year;
            this.month = options.month;
            this.day = options.day;
        }
    }
    serialize() {
        this.payload = Buffer.from([
            // 2 bytes placeholder for year
            0,
            0,
            this.month,
            this.day,
        ]);
        this.payload.writeUInt16BE(this.year, 0);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                date: `${strings_1.padStart(this.year.toString(), 4, "0")}-${strings_1.padStart(this.month.toString(), 2, "0")}-${strings_1.padStart(this.day.toString(), 2, "0")}`,
            },
        };
    }
};
TimeCCDateReport = __decorate([
    CommandClass_1.CCCommand(TimeCommand.DateReport)
], TimeCCDateReport);
exports.TimeCCDateReport = TimeCCDateReport;
let TimeCCDateGet = class TimeCCDateGet extends TimeCC {
};
TimeCCDateGet = __decorate([
    CommandClass_1.CCCommand(TimeCommand.DateGet),
    CommandClass_1.expectedCCResponse(TimeCCDateReport)
], TimeCCDateGet);
exports.TimeCCDateGet = TimeCCDateGet;
let TimeCCTimeOffsetSet = class TimeCCTimeOffsetSet extends TimeCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.standardOffset = options.standardOffset;
            this.dstOffset = options.dstOffset;
            this.dstStartDate = options.dstStart;
            this.dstEndDate = options.dstEnd;
        }
    }
    serialize() {
        const signTZO = Math.sign(this.standardOffset);
        const hourTZO = Math.floor(Math.abs(this.standardOffset) / 60);
        const hourByte = (signTZO < 0 ? 128 : 0) | (hourTZO & 127);
        const minuteTZO = Math.abs(this.standardOffset) % 60;
        const delta = this.dstOffset - this.standardOffset;
        const signDelta = Math.sign(delta);
        const minuteDelta = Math.abs(delta);
        const deltaByte = (signDelta < 0 ? 128 : 0) | (minuteDelta & 127);
        this.payload = Buffer.from([
            hourByte,
            minuteTZO,
            deltaByte,
            this.dstStartDate.getUTCMonth() + 1,
            this.dstStartDate.getUTCDate(),
            this.dstStartDate.getUTCHours(),
            this.dstEndDate.getUTCMonth() + 1,
            this.dstEndDate.getUTCDate(),
            this.dstEndDate.getUTCHours(),
        ]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "standard time offset": `${this.standardOffset} minutes`,
                "DST offset": `${this.dstOffset} minutes`,
                "DST start date": core_1.formatDate(this.dstStartDate, "YYYY-MM-DD"),
                "DST end date": core_1.formatDate(this.dstEndDate, "YYYY-MM-DD"),
            },
        };
    }
};
TimeCCTimeOffsetSet = __decorate([
    CommandClass_1.CCCommand(TimeCommand.TimeOffsetSet)
], TimeCCTimeOffsetSet);
exports.TimeCCTimeOffsetSet = TimeCCTimeOffsetSet;
let TimeCCTimeOffsetReport = class TimeCCTimeOffsetReport extends TimeCC {
    // @noCCValues Time is temporary :), we don't want to store that in a DB
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 9);
        // TODO: Refactor this into its own method
        const hourSign = !!(this.payload[0] & 128);
        const hour = this.payload[0] & 127;
        const minute = this.payload[1];
        this._standardOffset = (hourSign ? -1 : 1) * (hour * 60 + minute);
        const deltaSign = !!(this.payload[2] & 128);
        const deltaMinutes = this.payload[2] & 127;
        this._dstOffset =
            this._standardOffset + (deltaSign ? -1 : 1) * deltaMinutes;
        const currentYear = new Date().getUTCFullYear();
        this._dstStartDate = new Date(Date.UTC(currentYear, this.payload[3] - 1, this.payload[4], this.payload[5]));
        this._dstEndDate = new Date(Date.UTC(currentYear, this.payload[6] - 1, this.payload[7], this.payload[8]));
    }
    get standardOffset() {
        return this._standardOffset;
    }
    get dstOffset() {
        return this._dstOffset;
    }
    get dstStartDate() {
        return this._dstStartDate;
    }
    get dstEndDate() {
        return this._dstEndDate;
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "standard time offset": `${this._standardOffset} minutes`,
                "DST offset": `${this._dstOffset} minutes`,
                "DST start date": core_1.formatDate(this._dstStartDate, "YYYY-MM-DD"),
                "DST end date": core_1.formatDate(this._dstEndDate, "YYYY-MM-DD"),
            },
        };
    }
};
TimeCCTimeOffsetReport = __decorate([
    CommandClass_1.CCCommand(TimeCommand.TimeOffsetReport)
], TimeCCTimeOffsetReport);
exports.TimeCCTimeOffsetReport = TimeCCTimeOffsetReport;
let TimeCCTimeOffsetGet = class TimeCCTimeOffsetGet extends TimeCC {
};
TimeCCTimeOffsetGet = __decorate([
    CommandClass_1.CCCommand(TimeCommand.TimeOffsetGet),
    CommandClass_1.expectedCCResponse(TimeCCTimeOffsetReport)
], TimeCCTimeOffsetGet);
exports.TimeCCTimeOffsetGet = TimeCCTimeOffsetGet;

//# sourceMappingURL=TimeCC.js.map
