/// <reference types="node" />
import { ConfigManager } from "@zwave-js/config";
import { CommandClasses, Duration, LogConfig } from "@zwave-js/core";
import { DeepPartial } from "@zwave-js/shared";
import { EventEmitter } from "events";
import * as util from "util";
import { CommandClass } from "../commandclass/CommandClass";
import { SupervisionResult, SupervisionStatus } from "../commandclass/SupervisionCC";
import { ZWaveController } from "../controller/Controller";
import { SendDataRequest } from "../controller/SendDataMessages";
import { FunctionType, MessagePriority } from "../message/Constants";
import { Message } from "../message/Message";
import type { FileSystem } from "./FileSystem";
export interface ZWaveOptions {
    /** Specify timeouts in milliseconds */
    timeouts: {
        /** how long to wait for an ACK */
        ack: number;
        /** not sure */
        byte: number;
        /**
         * How long to wait for a controller response. Usually this timeout should never elapse,
         * so this is merely a safeguard against the driver stalling
         */
        response: number;
        /** How long to wait for a callback from the host for a SendData[Multicast]Request */
        sendDataCallback: number;
        /** How much time a node gets to process a request and send a response */
        report: number;
        /** How long generated nonces are valid */
        nonce: number;
    };
    attempts: {
        /** How often the driver should try communication with the controller before giving up */
        controller: number;
        /** How often the driver should try sending SendData commands before giving up */
        sendData: number;
        /** Whether a command should be retried when a node acknowledges the receipt but no response is received */
        retryAfterTransmitReport: boolean;
        /**
         * How many attempts should be made for each node interview before giving up
         */
        nodeInterview: number;
    };
    /**
     * Optional log configuration
     */
    logConfig?: LogConfig;
    storage: {
        /** Allows you to replace the default file system driver used to store and read the cache */
        driver: FileSystem;
        /** Allows you to specify a different cache directory */
        cacheDir: string;
        /**
         * How frequently the values and metadata should be written to the DB files. This is a compromise between data loss
         * in cause of a crash and disk wear:
         *
         * * `"fast"` immediately writes every change to disk
         * * `"slow"` writes at most every 5 minutes or after 500 changes - whichever happens first
         * * `"normal"` is a compromise between the two options
         */
        throttle: "fast" | "normal" | "slow";
    };
    /** Specify the network key to use for encryption. This must be a Buffer of exactly 16 bytes. */
    networkKey?: Buffer;
    /**
     * Some Command Classes support reporting that a value is unknown.
     * When this flag is `false`, unknown values are exposed as `undefined`.
     * When it is `true`, unknown values are exposed as the literal string "unknown" (even if the value is normally numeric).
     * Default: `false` */
    preserveUnknownValues?: boolean;
}
/**
 * Function signature for a message handler. The return type signals if the
 * message was handled (`true`) or further handlers should be called (`false`)
 */
export declare type RequestHandler<T extends Message = Message> = (msg: T) => boolean | Promise<boolean>;
export interface SendMessageOptions {
    /** The priority of the message to send. If none is given, the defined default priority of the message class will be used. */
    priority?: MessagePriority;
    /** If an exception should be thrown when the message to send is not supported. Setting this to false is is useful if the capabilities haven't been determined yet. Default: true */
    supportCheck?: boolean;
    /**
     * Whether the driver should update the node status to asleep or dead when a transaction is not acknowledged (repeatedly).
     * Setting this to false will cause the simply transaction to be rejected on failure.
     * Default: true
     */
    changeNodeStatusOnMissingACK?: boolean;
    /** Sets the number of milliseconds after which a message expires. When the expiration timer elapses, the promise is rejected with the error code `Controller_MessageExpired`. */
    expire?: number;
}
export interface SendCommandOptions extends SendMessageOptions {
    /** How many times the driver should try to send the message. Defaults to the configured Driver option */
    maxSendAttempts?: number;
}
export declare type SupervisionUpdateHandler = (status: SupervisionStatus, remainingDuration?: Duration) => void;
export declare type SendSupervisedCommandOptions = SendCommandOptions & ({
    requestStatusUpdates: false;
} | {
    requestStatusUpdates: true;
    onUpdate: SupervisionUpdateHandler;
});
export interface DriverEventCallbacks {
    "driver ready": () => void;
    "all nodes ready": () => void;
    error: (err: Error) => void;
}
export declare type DriverEvents = Extract<keyof DriverEventCallbacks, string>;
export interface Driver {
    on<TEvent extends DriverEvents>(event: TEvent, callback: DriverEventCallbacks[TEvent]): this;
    once<TEvent extends DriverEvents>(event: TEvent, callback: DriverEventCallbacks[TEvent]): this;
    removeListener<TEvent extends DriverEvents>(event: TEvent, callback: DriverEventCallbacks[TEvent]): this;
    off<TEvent extends DriverEvents>(event: TEvent, callback: DriverEventCallbacks[TEvent]): this;
    removeAllListeners(event?: DriverEvents): this;
    emit<TEvent extends DriverEvents>(event: TEvent, ...args: Parameters<DriverEventCallbacks[TEvent]>): boolean;
}
/**
 * The driver is the core of this library. It controls the serial interface,
 * handles transmission and receipt of messages and manages the network cache.
 * Any action you want to perform on the Z-Wave network must go through a driver
 * instance or its associated nodes.
 */
export declare class Driver extends EventEmitter {
    private port;
    /** The serial port instance */
    private serial;
    /** An instance of the Send Thread state machine */
    private sendThread;
    /** A map of handlers for all sorts of requests */
    private requestHandlers;
    /** A map of awaited commands */
    private awaitedCommands;
    /** A map of all current supervision sessions that may still receive updates */
    private supervisionSessions;
    readonly cacheDir: string;
    private _valueDB;
    private _metadataDB;
    readonly configManager: ConfigManager;
    private _logContainer;
    private _driverLog;
    private _controllerLog;
    private _controller;
    /** Encapsulates information about the Z-Wave controller and provides access to its nodes */
    get controller(): ZWaveController;
    private _securityManager;
    constructor(port: string, options?: DeepPartial<ZWaveOptions>);
    /** Updates the logging configuration without having to restart the driver. */
    updateLogConfig(config: DeepPartial<LogConfig>): void;
    /** Returns the current logging configuration. */
    getLogConfig(): LogConfig;
    /** Enumerates all existing serial ports */
    static enumerateSerialPorts(): Promise<string[]>;
    private _wasStarted;
    private _isOpen;
    /** Start the driver */
    start(): Promise<void>;
    private _controllerInterviewed;
    private _nodesReady;
    private _nodesReadyEventEmitted;
    /** Indicates whether all nodes are ready, i.e. the "all nodes ready" event has been emitted */
    get allNodesReady(): boolean;
    /**
     * Initializes the variables for controller and nodes,
     * adds event handlers and starts the interview process.
     */
    private initializeControllerAndNodes;
    private retryNodeInterviewTimeouts;
    /** Adds the necessary event handlers for a node instance */
    private addNodeEventHandlers;
    /** Removes a node's event handlers that were added with addNodeEventHandlers */
    private removeNodeEventHandlers;
    /** Is called when a node wakes up */
    private onNodeWakeUp;
    /** Is called when a node goes to sleep */
    private onNodeSleep;
    /** Is called when a previously dead node starts communicating again */
    private onNodeAlive;
    /** Is called when a node is marked as dead */
    private onNodeDead;
    /** Is called when a node is ready to be used */
    private onNodeReady;
    /** Checks if all nodes are ready and emits the "all nodes ready" event if they are */
    private checkAllNodesReady;
    /** Is called when a node interview is completed */
    private onNodeInterviewCompleted;
    /** This is called when a new node has been added to the network */
    private onNodeAdded;
    /** This is called when a node was removed from the network */
    private onNodeRemoved;
    /** This is called when a node's firmware was updated */
    private onNodeFirmwareUpdated;
    /** Checks if there are any pending messages for the given node */
    private hasPendingMessages;
    /**
     * Retrieves the maximum version of a command class the given endpoint supports.
     * Returns 0 when the CC is not supported. Also returns 0 when the node was not found.
     * Falls back to querying the root endpoint if an endpoint was not found on the node
     *
     * @param cc The command class whose version should be retrieved
     * @param nodeId The node for which the CC version should be retrieved
     * @param endpointIndex The endpoint in question
     */
    getSupportedCCVersionForEndpoint(cc: CommandClasses, nodeId: number, endpointIndex?: number): number;
    /**
     * Retrieves the maximum version of a command class that can be used to communicate with a node.
     * Returns 1 if the node claims that it does not support a CC.
     * Throws if the CC is not implemented in this library yet.
     *
     * @param cc The command class whose version should be retrieved
     * @param nodeId The node for which the CC version should be retrieved
     * @param endpointIndex The endpoint for which the CC version should be retrieved
     */
    getSafeCCVersionForNode(cc: CommandClasses, nodeId: number, endpointIndex?: number): number;
    /**
     * Performs a hard reset on the controller. This wipes out all configuration!
     *
     * The returned Promise resolves when the hard reset has been performed.
     * It does not wait for the initialization process which is started afterwards.
     */
    hardReset(): Promise<void>;
    private _wasDestroyed;
    /**
     * Ensures that the driver is ready to communicate (serial port open and not destroyed).
     * If desired, also checks that the controller interview has been completed.
     */
    private ensureReady;
    /** Indicates whether the driver is ready, i.e. the "driver ready" event was emitted */
    get ready(): boolean;
    private _cleanupHandler;
    /**
     * Terminates the driver instance and closes the underlying serial connection.
     * Must be called under any circumstances.
     */
    destroy(): Promise<void>;
    private serialport_onError;
    /**
     * Is called when the serial port has received a single-byte message or a complete message buffer
     */
    private serialport_onData;
    /** Handles a decoding error and returns the desired reply to the stick */
    private handleDecodeError;
    /** Checks if a transaction failed because a node didn't respond in time */
    private isMissingNodeACK;
    /**
     * Handles the case that a node failed to respond in time.
     * Returns `true` if the transaction failure was handled, `false` if it needs to be rejected.
     */
    private handleMissingNodeACK;
    private partialCCSessions;
    /**
     * Assembles partial CCs of in a message body. Returns `true` when the message is complete and can be handled further.
     * If the message expects another partial one, this returns `false`.
     */
    private assemblePartialCCs;
    /**
     * Is called when a message is received that does not belong to any ongoing transactions
     * @param msg The decoded message
     */
    private handleUnsolicitedMessage;
    /**
     * Registers a handler for messages that are not handled by the driver as part of a message exchange.
     * The handler function needs to return a boolean indicating if the message has been handled.
     * Registered handlers are called in sequence until a handler returns `true`.
     *
     * @param fnType The function type to register the handler for
     * @param handler The request handler callback
     * @param oneTime Whether the handler should be removed after its first successful invocation
     */
    registerRequestHandler<T extends Message>(fnType: FunctionType, handler: RequestHandler<T>, oneTime?: boolean): void;
    /**
     * Unregisters a message handler that has been added with `registerRequestHandler`
     * @param fnType The function type to unregister the handler for
     * @param handler The previously registered request handler callback
     */
    unregisterRequestHandler(fnType: FunctionType, handler: RequestHandler): void;
    /**
     * Checks whether a CC may be handled or should be ignored.
     * This method expects `cc` to be unwrapped.
     */
    private mayHandleUnsolicitedCommand;
    /**
     * Is called when a Request-type message was received
     */
    private handleRequest;
    private lastCallbackId;
    /**
     * Returns the next callback ID. Callback IDs are used to correllate requests
     * to the controller/nodes with its response
     */
    getNextCallbackId(): number;
    private encapsulateCommands;
    private unwrapCommands;
    /**
     * Sends a message to the Z-Wave stick.
     * @param msg The message to send
     * @param options (optional) Options regarding the message transmission
     */
    sendMessage<TResponse extends Message = Message>(msg: Message, options?: SendMessageOptions): Promise<TResponse>;
    /**
     * Sends a command to a Z-Wave node. If the node returns a command in response, that command will be the return value.
     * If the command expects no response **or** the response times out, nothing will be returned
     * @param command The command to send. It will be encapsulated in a SendData[Multicast]Request.
     * @param options (optional) Options regarding the message transmission
     */
    sendCommand<TResponse extends CommandClass = CommandClass>(command: CommandClass, options?: SendCommandOptions): Promise<TResponse | undefined>;
    /**
     * Sends a supervised command to a Z-Wave node. When status updates are requested, the passed callback will be executed for every non-final update.
     * @param command The command to send
     * @param options (optional) Options regarding the message transmission
     */
    sendSupervisedCommand(command: CommandClass, options?: SendSupervisedCommandOptions): Promise<SupervisionResult | undefined>;
    /**
     * Sends a supervised command to a Z-Wave node if the Supervision CC is supported. If not, a normal command is sent.
     * This does not return any Report values, so only use this for Set-type commands.
     *
     * @param command The command to send
     * @param options (optional) Options regarding the message transmission
     */
    trySendCommandSupervised(command: CommandClass, options?: SendSupervisedCommandOptions): Promise<SupervisionResult | undefined>;
    /**
     * Sends a low-level message like ACK, NAK or CAN immediately
     * @param message The low-level message to send
     */
    private writeHeader;
    /** Sends a raw datagram to the serialport (if that is open) */
    private writeSerial;
    /**
     * Waits until a command is received or a timeout has elapsed. Returns the received command.
     * @param timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected
     * @param predicate A predicate function to test all incoming command classes
     */
    waitForCommand<T extends CommandClass>(predicate: (cc: CommandClass) => boolean, timeout: number): Promise<T>;
    /** Checks if a message is allowed to go into the wakeup queue */
    private mayMoveToWakeupQueue;
    /** Moves all messages for a given node into the wakeup queue */
    private moveMessagesToWakeupQueue;
    /** Re-sorts the send queue */
    private sortSendQueue;
    private lastSaveToCache;
    private readonly saveToCacheInterval;
    private saveToCacheTimer;
    private isSavingToCache;
    /**
     * Does the work for saveNetworkToCache. This is not throttled, so any call
     * to this method WILL save the network.
     */
    private saveNetworkToCacheInternal;
    /**
     * Saves the current configuration and collected data about the controller and all nodes to a cache file.
     * For performance reasons, these calls may be throttled.
     */
    saveNetworkToCache(): Promise<void>;
    /**
     * Restores a previously stored Z-Wave network state from cache to speed up the startup process
     */
    restoreNetworkStructureFromCache(): Promise<void>;
    private sendNodeToSleepTimers;
    /** Computes the maximum net CC payload size for the given CC or SendDataRequest */
    computeNetCCPayloadSize(commandOrMsg: CommandClass | SendDataRequest): number;
    [util.inspect.custom](): string;
}
